#=====================================================================================================================
# Script TCL para Eggdrop con ChatGPT (OpenAI) vía Python
# El script responde solo cuando mencionan el nick del bot dentro del canal
# Se limita la cantidad de líneas (20) al canal; si es demasiado largo, redirige a una web de IA (Si cuentas con ella)
# Probado en Eggdrop 1.10.x - (Tcl 8.x) - Python3.8
# Autor: charls_a
# E-mail: charls_a@canal-ayuda.com - https://canal-ayuda.com
# Versión: 1.0
# Reporte de bugs: #Ayuda @ Undernet.org
# RECUERDA CONFIGURAR TODAS LAS VARIABLES PARA QUE EL BOT PUEDA FUNCIONAR CORRECTAMENTE
# ====================================================================================================================

# ========== CONFIGURACIÓN ==========

# Ruta del directorio dónde se ejecuta Python 3 (Verificar ruta)
set ::chatgpt(python) "/usr/bin/python3"

# Ruta del directorio donde se encuentra ubicado el script Python
set ::chatgpt(script) "/mi/ruta/de/directorio/chatgpt.py"

# Directorio de trabajo (donde se ejecutará el Python)
# Si lo dejas vacío, no hace cd.
set ::chatgpt(workdir) "/mi/ruta/de/directorio/chatgpt"

# API key de OpenAI
set ::chatgpt(api_key) "TU_CLAVE_API_DE_OPENAI"

# Modelo de OpenAI
# Modelos disponibles: gpt-3.5-turbo, gpt-3.5-turbo-16k, text-davinci-003, text-davinci-002, gpt-4, gpt-4-turbo, gpt-4-32k,
# gpt-4o, gpt-4o-mini, gpt-4.1, gpt-4.1-mini, gpt-4.1-nano, gpt-5, gpt-oss-20b, gpt-oss-120b
set ::chatgpt(model) "gpt-4.1-mini"

# Máximo de tokens de salida que se solicitarán a ChatGPT
set ::chatgpt(max_tokens) 1200

# Número máximo de caracteres a enviar al canal por línea
# (valores bajos evitan cortes por límite de IRC)
set ::chatgpt(max_chars) 200

# Estilo de comportamiento:
# informativo, profesional, creativo, educativo, divertido
set ::chatgpt(style) "profesional"

# Lista de canales donde el bot responderá
# Configura esto por tus canales reales, ejemplo: {#canal1 #canal2}
set ::chatgpt(channels) {#canal1}

# --- CONFIGURACIÓN DE CONTEXTO (MEMORIA POR NICK) ---

# Activar/desactivar contexto (0 = sin contexto, más rápido)
set ::chatgpt(context_enabled) 1

# Tiempo máximo (en segundos) que se considera una "misma conversación".
set ::chatgpt(context_timeout) 900

# Máximo de caracteres de historial que se guardan por usuario
set ::chatgpt(context_max_chars) 1500

# Diccionario donde se guarda el contexto:
# key = nick en minúsculas
# valor = {timestamp_unix historial}
set ::chatgpt(context) [dict create]

# --- LOG DE ERRORES ---

# Archivo donde se guardarán los errores del bot
set ::chatgpt(log_file) "/mi/ruta/de/directorio/chatgpt.log"

# ¿Registrar errores? 1 = sí, 0 = no
set ::chatgpt(logging_enabled) 1

# --- LÍMITES DE SALIDA ---

# Máximo de líneas que el bot enviará al canal
set ::chatgpt(max_lines) 20

# URL de la página de IA para consultas largas (Habilita esta línea, si tienes un sitio web implementado con IA)
#set ::chatgpt(redirect_url) "https://mi-sitio-web-con-ia.com"

# ========== LOG ERROR ==========

proc chatgpt:log_error {msg} {
    if {!$::chatgpt(logging_enabled)} {
        return
    }
    set ts [clock format [clock seconds] -format "%Y-%m-%d %H:%M:%S"]
    set entry "[$ts] $msg\n"

    if {[catch {
        set fp [open $::chatgpt(log_file) "a"]
        puts -nonewline $fp $entry
        close $fp
    } err]} {
        putlog "ChatGPT: ERROR: No se pudo escribir en chatgpt.log ($err)"
    }
}

# ========== VERIFICACIONES INICIALES ==========

proc chatgpt:check_config {} {
    if {[lsearch -exact [info commands] exec] == -1} {
        set m "ChatGPT: ATENCIÓN: el comando 'exec' no está disponible en este Eggdrop."
        putlog $m
        chatgpt:log_error $m
    }

    if {![file executable $::chatgpt(python)]} {
        set m "ChatGPT: ATENCIÓN: Python no encontrado o no ejecutable en $::chatgpt(python)."
        putlog $m
        chatgpt:log_error $m
    }

    if {![file readable $::chatgpt(script)]} {
        set m "ChatGPT: ATENCIÓN: script Python no legible: $::chatgpt(script)."
        putlog $m
        chatgpt:log_error $m
    }

    if {$::chatgpt(workdir) ne "" && ![file isdirectory $::chatgpt(workdir)]} {
        set m "ChatGPT: ATENCIÓN: workdir no existe o no es directorio: $::chatgpt(workdir)."
        putlog $m
        chatgpt:log_error $m
    }

    if {[llength $::chatgpt(channels)] == 0} {
        set m "ChatGPT: ATENCIÓN: lista de canales vacía, el bot no responderá en ningún canal."
        putlog $m
        chatgpt:log_error $m
    }

    if {$::chatgpt(api_key) eq ""} {
        set m "ChatGPT: aviso: api_key no configurada en Tcl, se espera OPENAI_API_KEY en el entorno."
        putlog $m
        chatgpt:log_error $m
    }
}
chatgpt:check_config

# ========== UTILIDADES ==========

# Recorta una línea a un máximo de caracteres
proc chatgpt:shorten_line {text max} {
    if {[string length $text] > $max} {
        return [string range $text 0 [expr {$max - 1}]]
    } else {
        return $text
    }
}

# Limpia el texto para que sea amigable al IRC:
# - elimina backticks (`) y dobles asteriscos (**) usados en Markdown
proc chatgpt:cleanup_text {text} {
    regsub -all {`} $text "" text
    regsub -all {\*\*} $text "" text
    return $text
}

# Convierte el texto completo en una lista de "chunks" listos para enviar
proc chatgpt:chunk_text {text max} {
    set chunks {}
    set text [chatgpt:cleanup_text $text]
    set lines [split $text "\n"]

    foreach line $lines {
        set line [string trim $line]
        if {$line eq ""} {
            continue
        }

        while {[string length $line] > 0} {
            set chunk [chatgpt:shorten_line $line $max]
            lappend chunks $chunk

            if {[string length $line] > [string length $chunk]} {
                set line [string range $line [string length $chunk] end]
                set line [string trimleft $line]
            } else {
                set line ""
            }
        }
    }

    return $chunks
}

# Envía texto ya troceado (chunks) al canal
proc chatgpt:send_chunks {chan chunks} {
    foreach chunk $chunks {
        puthelp "PRIVMSG $chan :$chunk"
    }
}

# ========== MANEJO DE CONTEXTO Y PROMPT ==========

# Construye el prompt a enviar a ChatGPT, limpiando el nick del bot y,
# si está activo, añadiendo contexto por usuario.
proc chatgpt:build_prompt {nick user_text} {
    global botnick

    # Partimos del texto original
    set clean $user_text

    # Eliminar "botnick:" o "botnick," al inicio de la línea (mención típica)
    # Ej: "carlos_a: qué modelo eres" → "qué modelo eres"
    regsub -nocase "^${botnick}(:|,| )+" $clean "" clean

    # Eliminar menciones del bot en medio del texto
    # Ej: "hola carlos_a qué modelo eres" → "hola  qué modelo eres"
    regsub -nocase "${botnick}" "" clean

    # Limpiar espacios extra
    set clean [string trim $clean]

    # Si no usamos contexto, prompt simple
    if {!$::chatgpt(context_enabled)} {
        return "El usuario hace una pregunta en un canal de IRC. Responde directamente a la pregunta, sin saludar, sin mencionar nicks y sin usar formato Markdown. Pregunta: $clean"
    }

    # --- MODO CON CONTEXTO ---
    set now [clock seconds]
    set key [string tolower $nick]
    set history ""

    if {[dict exists $::chatgpt(context) $key]} {
        lassign [dict get $::chatgpt(context) $key] ts hist
        if {$now - $ts <= $::chatgpt(context_timeout)} {
            set history $hist
        } else {
            set ::chatgpt(context) [dict remove $::chatgpt(context) $key]
        }
    }

    if {$history eq ""} {
        return "Conversación en un canal de IRC. Responde sin saludar, sin mencionar nicks y sin usar formato Markdown. Pregunta actual: $clean"
    } else {
        return "Continuación de una conversación en un canal de IRC. No saludes ni menciones nicks y no uses formato Markdown. Historial resumido:\n$history\nPregunta actual: $clean"
    }
}

proc chatgpt:update_context {nick user_text answer} {
    if {!$::chatgpt(context_enabled)} {
        return
    }

    set now [clock seconds]
    set key [string tolower $nick]

    set new_hist "Usuario ($nick): $user_text\nBot: $answer"
    set combined $new_hist

    if {[dict exists $::chatgpt(context) $key]} {
        lassign [dict get $::chatgpt(context) $key] ts old_hist
        set combined "$old_hist\n$new_hist"
    }

    if {[string length $combined] > $::chatgpt(context_max_chars)} {
        set start_index [expr {[string length $combined] - $::chatgpt(context_max_chars)}]
        set combined [string range $combined $start_index end]
    }

    set ::chatgpt(context) [dict replace $::chatgpt(context) $key [list $now $combined]]
}

# ========== BIND: MENCIÓN DEL BOT EN CANAL ==========

bind pubm - * chatgpt:pubm_handler

proc chatgpt:pubm_handler {nick uhost hand chan text} {
    global botnick

    # Solo responder en canales configurados
    if {[lsearch -exact $::chatgpt(channels) $chan] == -1} {
        return
    }

    # Evitar responderse a sí mismo
    if {[string equal -nocase $nick $botnick]} {
        return
    }

    # Comprobar mención del bot en el texto
    set lowtext [string tolower $text]
    set lownick [string tolower $botnick]
    if {[string first $lownick $lowtext] == -1} {
        return
    }

    # Construir prompt con o sin contexto
    set prompt [chatgpt:build_prompt $nick $text]

    # Construir comando para exec
    set cmd [list $::chatgpt(python) $::chatgpt(script) \
                 --model $::chatgpt(model) \
                 --max-tokens $::chatgpt(max_tokens) \
                 --prompt $prompt]

    if {$::chatgpt(api_key) ne ""} {
        lappend cmd --api-key $::chatgpt(api_key)
    }
    if {$::chatgpt(style) ne ""} {
        lappend cmd --style $::chatgpt(style)
    }

    set result ""

    # Ejecutar el script Python en el directorio de trabajo (si está configurado)
    if {$::chatgpt(workdir) ne "" && [file isdirectory $::chatgpt(workdir)]} {
        set oldpwd [pwd]
        cd $::chatgpt(workdir)
        set rc [catch {eval exec $cmd} result]
        cd $oldpwd
    } else {
        set rc [catch {eval exec $cmd} result]
    }

    if {$rc} {
        set errmsg "Error ejecutando ChatGPT para $nick en $chan: $result"
        putlog $errmsg
        chatgpt:log_error $errmsg
        puthelp "PRIVMSG $chan :Ocurrió un error hablando con ChatGPT."
        return
    }

    set answer [string trim $result]
    if {$answer eq ""} {
        chatgpt:log_error "Respuesta vacía recibida desde ChatGPT para usuario $nick."
        set answer "No estoy muy seguro de qué responder a eso, ¿me lo puedes explicar un poquito más?"
    }

    # Actualizar contexto si está activado
    chatgpt:update_context $nick $text $answer

    # Preparar trozos y aplicar límite de líneas
    set chunks [chatgpt:chunk_text $answer $::chatgpt(max_chars)]
    set total  [llength $chunks]

    if {$total > $::chatgpt(max_lines)} {
        set msg "Esta consulta requiere una explicación más extensa de lo permitido en el canal. Puedes hacerla completa aquí: $::chatgpt(redirect_url)"
        puthelp "PRIVMSG $chan :$msg"
    } else {
        chatgpt:send_chunks $chan $chunks
    }
}
putlog ">> ChatGPT TCL v1.0 cargado correctamente ✅"
putlog ">> Creado por charls_a | https://canal-ayuda.com"
