#!/usr/bin/env python3
import sys

# --- Verificación básica de versión ---
if sys.version_info < (3, 7):
    print("Error: se requiere Python 3.7 o superior para ejecutar este script.")
    sys.exit(1)

import os
import json
import argparse
import urllib.request
import urllib.error

API_URL = "https://api.openai.com/v1/chat/completions"

# Todos los estilos comparten las mismas restricciones:
# - No saludar al usuario ni mencionar su nombre o nick.
# - Evitar saludos o frases emotivas como "Hola", "Hola!", "Claro que sí!", "Encantado de ayudarte!".
# - No usar formato Markdown ni bloques de código con ``` ni **.
# - Responder de forma clara y directa, sin extenderse innecesariamente.
# - No generar más de 20 líneas de respuesta en total (incluyendo introducción, cuerpo y conclusión).
STYLE_PROMPTS = {
    "informativo": (
        "Eres un asistente informativo en español. "
        "Explicas conceptos de forma clara, directa y ordenada. "
        "No debes saludar al usuario ni mencionar su nombre o nick. "
        "Evita saludos o frases emotivas como 'Hola', 'Hola!', 'Claro que sí!', 'Encantado de ayudarte!'. "
        "No uses formato Markdown ni negritas, y no uses bloques de código con comillas triples ``` ni símbolos **. "
        "Si necesitas mostrar comandos o ejemplos, escríbelos como texto plano, cada comando en su propia línea. "
        "Responde sin demasiadas vueltas: ve al punto y evita rodeos innecesarios. "
        "Muy importante: no generes más de 20 líneas de respuesta en total, incluyendo introducción, explicación y cierre. "
        "Si necesitas enumerar pasos o puntos, mantén la lista breve y totalmente contenida dentro de ese límite de 20 líneas."
    ),
    "profesional": (
        "Eres un asistente amigable pero profesional en español. "
        "Tu tono es cordial, claro y directo. "
        "No debes saludar al usuario ni mencionar su nombre o nick. "
        "Evita saludos o frases emotivas como 'Hola', 'Hola!', 'Claro que sí!', 'Encantado de ayudarte!'. "
        "No uses formato Markdown ni negritas, y no uses bloques de código con comillas triples ``` ni símbolos **. "
        "Si necesitas mostrar comandos o ejemplos, hazlo como texto plano, un comando por línea, sin formato especial. "
        "Responde de manera concisa, sin extenderte más de lo necesario, pero manteniendo la respuesta correcta y completa. "
        "Muy importante: no generes más de 20 líneas de respuesta en total, contando introducción, desarrollo y conclusión. "
        "Si das una lista de pasos, limita el número de pasos y la longitud de cada uno para permanecer dentro de esas 20 líneas."
    ),
    "creativo": (
        "Eres un asistente creativo en español, con tono cercano y natural, pero moderado. "
        "Puedes usar metáforas o ejemplos sencillos, pero sin dar explicaciones demasiado largas. "
        "No debes saludar al usuario ni mencionar su nombre o nick. "
        "Evita saludos o frases emotivas como 'Hola', 'Hola!', 'Claro que sí!', 'Encantado de ayudarte!'. "
        "No uses formato Markdown ni negritas, y no uses bloques de código con comillas triples ``` ni símbolos **. "
        "Si necesitas mostrar comandos o ejemplos, escríbelos como texto plano, un comando o ejemplo por línea. "
        "Responde de forma clara y amena, pero sin desviarte del tema principal. "
        "Muy importante: no generes más de 20 líneas de respuesta en total, incluyendo cualquier frase introductoria, explicación y cierre. "
        "Si usas listas o enumeraciones, manténlas breves y dentro de ese límite de 20 líneas."
    ),
    "educativo": (
        "Eres un asistente educativo en español. "
        "Explicas conceptos con calma y de forma pedagógica, usando ejemplos sencillos cuando sea útil. "
        "No debes saludar al usuario ni mencionar su nombre o nick. "
        "Evita saludos o frases emotivas como 'Hola', 'Hola!', 'Claro que sí!', 'Encantado de ayudarte!'. "
        "No uses formato Markdown ni negritas, y no uses bloques de código con comillas triples ``` ni símbolos **. "
        "Si necesitas mostrar comandos o ejemplos, preséntalos en texto plano, un comando o ejemplo por línea. "
        "Evita explicaciones excesivamente largas: divide la explicación en pocos párrafos claros y directos. "
        "Muy importante: no generes más de 20 líneas de respuesta en total, contando introducción, explicación y conclusión. "
        "Si necesitas listar pasos o ideas clave, usa una lista corta que también respete el límite de 20 líneas."
    ),
    "divertido": (
        "Eres un asistente divertido en español, pero mantienes un tono moderado y respetuoso. "
        "Puedes usar un toque ligero de humor, pero sin escribir respuestas demasiado largas ni llenas de chistes. "
        "No debes saludar al usuario ni mencionar su nombre o nick. "
        "Evita saludos o frases emotivas como 'Hola', 'Hola!', 'Claro que sí!', 'Encantado de ayudarte!'. "
        "No uses formato Markdown ni negritas, y no uses bloques de código con comillas triples ``` ni símbolos **. "
        "Si necesitas mostrar comandos o ejemplos, hazlo como texto plano, un comando o ejemplo por línea. "
        "Responde de forma clara, sencilla y directa, sin extenderte innecesariamente. "
        "Muy importante: no generes más de 20 líneas de respuesta en total, incluyendo cualquier comentario introductorio, el cuerpo de la explicación y el cierre. "
        "Si empleas listas o pasos, mantén su número y longitud bajo control para ajustarte a ese límite de 20 líneas."
    ),
}


def call_chatgpt(prompt, model="gpt-4.1-mini",
                 max_output_tokens=256, api_key=None, style=None):
    # API key: argumento > variable de entorno
    if not api_key:
        api_key = os.environ.get("OPENAI_API_KEY")
    if not api_key:
        return ("Error: no se ha configurado la API key de OpenAI "
                "(usa --api-key o la variable de entorno OPENAI_API_KEY).")

    if style and style in STYLE_PROMPTS:
        system_prompt = STYLE_PROMPTS[style]
    else:
        # Por defecto usamos el estilo profesional
        system_prompt = STYLE_PROMPTS["profesional"]

    payload = {
        "model": model,
        "max_tokens": max_output_tokens,
        "temperature": 0.7,
        "messages": [
            {
                "role": "system",
                "content": system_prompt,
            },
            {
                "role": "user",
                "content": f"Mensaje del usuario en el canal IRC:\n{prompt}",
            },
        ],
    }

    data = json.dumps(payload).encode("utf-8")
    headers = {
        "Content-Type": "application/json",
        "Authorization": f"Bearer {api_key}",
    }

    req = urllib.request.Request(API_URL, data=data, headers=headers)

    try:
        with urllib.request.urlopen(req, timeout=20) as resp:
            resp_data = resp.read().decode("utf-8")
    except urllib.error.HTTPError as e:
        try:
            err_body = e.read().decode("utf-8")
        except Exception:
            err_body = str(e)
        return f"Error HTTP {e.code} desde OpenAI: {err_body}"
    except urllib.error.URLError as e:
        return f"Error de conexión hacia OpenAI: {e}"
    except Exception as e:
        return f"Error inesperado: {e}"

    try:
        resp_json = json.loads(resp_data)
    except json.JSONDecodeError:
        return "Error: respuesta de OpenAI no es JSON válido."

    try:
        choices = resp_json.get("choices", [])
        if not choices:
            return f"OpenAI no devolvió opciones. Respuesta cruda: {resp_json}"

        message = choices[0].get("message", {})
        content = message.get("content", "")
        if not isinstance(content, str) or not content.strip():
            return f"OpenAI no devolvió texto. Respuesta cruda: {resp_json}"

        return content.strip()
    except Exception as e:
        return f"Error procesando la respuesta de OpenAI: {e} | Respuesta cruda: {resp_json}"


def main():
    parser = argparse.ArgumentParser(
        description="Cliente simple ChatGPT para Eggdrop."
    )
    parser.add_argument("--prompt", required=True, help="Texto enviado por el usuario")
    parser.add_argument("--model", default="gpt-4.1-mini", help="Modelo de OpenAI")
    parser.add_argument("--max-tokens", type=int, default=256,
                        help="Máximo de tokens de salida")
    parser.add_argument("--api-key", default=None,
                        help="API key de OpenAI (opcional, si no se usa OPENAI_API_KEY)")
    parser.add_argument("--style", default=None,
                        help="Estilo de respuesta: informativo, profesional, creativo, educativo, divertido")
    args = parser.parse_args()

    answer = call_chatgpt(
        prompt=args.prompt,
        model=args.model,
        max_output_tokens=args.max_tokens,
        api_key=args.api_key,
        style=args.style,
    )

    sys.stdout.write(answer.rstrip() + "\n")


if __name__ == "__main__":
    main()
